<?php


// ─── CONFIGURATION ──────────────────────────────────────────────────────────────

// Path to store the blockchain (must be writable by PHP)
define('CHAIN_FILE', __DIR__ . '/chain.json');

// AES‐256‑CBC encryption key (32 bytes). Change this to your own random string.
define('ENCRYPTION_KEY', 'YourVeryStrongSecretKey12345678!'); // exactly 32 bytes

// Base URL where you’ll host view_block.php (no trailing slash).
// E.g., 'https://yourdomain.com/path/to/scripts'
define('BASE_URL', 'http://localhost'); 

// ─── ENCRYPTION HELPERS ─────────────────────────────────────────────────────────

function encryptPayload(string $plaintext): array
{
    // Generate a random IV
    $iv = random_bytes(16);
    $cipher = openssl_encrypt($plaintext, 'AES-256-CBC', ENCRYPTION_KEY, OPENSSL_RAW_DATA, $iv);
    $combined = base64_encode($iv . $cipher);
    return [
        'encrypted_data' => $combined,
        'iv_base64'      => base64_encode($iv),
    ];
}

function decryptPayload(string $encrypted_combined): string
{
    $decoded = base64_decode($encrypted_combined);
    $iv      = substr($decoded, 0, 16);
    $cipher  = substr($decoded, 16);
    $plaintext = openssl_decrypt($cipher, 'AES-256-CBC', ENCRYPTION_KEY, OPENSSL_RAW_DATA, $iv);
    return $plaintext === false ? '' : $plaintext;
}

// ─── BLOCK & BLOCKCHAIN CLASSES ─────────────────────────────────────────────────

class Block
{
    public $index;
    public $timestamp;
    public $data;         // base64‐encoded encrypted JSON
    public $iv;           // base64‐encoded IV used for encryption
    public $previousHash;
    public $hash;
    
    public function __construct(int $index, string $timestamp, string $encryptedData, string $ivBase64, string $previousHash)
    {
        $this->index        = $index;
        $this->timestamp    = $timestamp;
        $this->data         = $encryptedData;
        $this->iv           = $ivBase64;
        $this->previousHash = $previousHash;
        $this->hash         = $this->calculateHash();
    }
    
    // Recompute this block’s hash (over its metadata and encrypted `data`)
    public function calculateHash(): string
    {
        $blockString = json_encode([
            'index'        => $this->index,
            'timestamp'    => $this->timestamp,
            'data'         => $this->data,
            'iv'           => $this->iv,
            'previousHash' => $this->previousHash
        ], JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE);
        
        return hash('sha256', $blockString);
    }
    
    // Return an associative array version (for JSON storage)
    public function toArray(): array
    {
        return [
            'index'        => $this->index,
            'timestamp'    => $this->timestamp,
            'data'         => $this->data,
            'iv'           => $this->iv,
            'previousHash' => $this->previousHash,
            'hash'         => $this->hash
        ];
    }
    
    // Build a Block object from an array (as read from JSON)
    public static function fromArray(array $arr): Block
    {
        $block = new self(
            $arr['index'],
            $arr['timestamp'],
            $arr['data'],
            $arr['iv'],
            $arr['previousHash']
        );
        // Overwrite the calculated hash with the stored one (to preserve chain)
        $block->hash = $arr['hash'];
        return $block;
    }
}

class Blockchain
{
    /** @var Block[] */
    public $chain = [];
    
    public function __construct()
    {
        // Try loading existing chain; if none, create with genesis block
        if (file_exists(CHAIN_FILE)) {
            $this->loadChain();
        } else {
            $this->chain[] = $this->createGenesisBlock();
            $this->saveChain();
        }
    }
    
    private function createGenesisBlock(): Block
    {
        $timestamp = date('Y-m-d H:i:s');
        $data      = encryptPayload("Genesis Block")[ 'encrypted_data' ];
        $iv        = encryptPayload("Genesis Block")[ 'iv_base64' ];
        return new Block(0, $timestamp, $data, $iv, '0');
    }
    
    public function getLatestBlock(): Block
    {
        return $this->chain[count($this->chain) - 1];
    }
    
    /**
     * Add a new “transaction” to the chain.
     * @param string $type         e.g., 'add_voter', 'add_candidate', 'add_admin', etc.
     * @param array  $details      Arbitrary associative array with the payload (e.g., ['fname'=>'John', 'lname'=>'Doe', ...])
     */
    public function addTransaction(string $type, array $details): Block
    {
        $latest      = $this->getLatestBlock();
        $newIndex    = count($this->chain);
        $timestamp   = date('Y-m-d H:i:s');
        
        // Build a plain‐text JSON payload
        $payload = json_encode([
            'type'      => $type,
            'timestamp' => $timestamp,
            'details'   => $details
        ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
        
        // Encrypt payload
        $encArr   = encryptPayload($payload);
        $encryptedData = $encArr['encrypted_data'];
        $ivBase64      = $encArr['iv_base64'];
        
        // Create and append the new block
        $newBlock = new Block($newIndex, $timestamp, $encryptedData, $ivBase64, $latest->hash);
        $this->chain[] = $newBlock;
        
        // Persist to disk
        $this->saveChain();
        
        return $newBlock;
    }
    
    /** Verify the integrity of the entire chain */
    public function isChainValid(): bool
    {
        for ($i = 1; $i < count($this->chain); $i++) {
            $current  = $this->chain[$i];
            $previous = $this->chain[$i - 1];
            
            if ($current->hash !== $current->calculateHash()) {
                return false;
            }
            if ($current->previousHash !== $previous->hash) {
                return false;
            }
        }
        return true;
    }
    
    /** Save the chain array to chain.json */
    private function saveChain(): void
    {
        $arr = [];
        foreach ($this->chain as $block) {
            $arr[] = $block->toArray();
        }
        file_put_contents(CHAIN_FILE, json_encode($arr, JSON_PRETTY_PRINT));
    }
    
    /** Load the chain from chain.json */
    private function loadChain(): void
    {
        $raw = file_get_contents(CHAIN_FILE);
        $arr = json_decode($raw, true);
        $this->chain = [];
        foreach ($arr as $blockArr) {
            $this->chain[] = Block::fromArray($blockArr);
        }
    }
    
    /**
     * Generate a URL where someone can view this block’s decrypted contents.
     * @param Block $block
     * @return string
     */
    public function getBlockLink(Block $block): string
    {
        return BASE_URL . '/view_block.php?index=' . $block->index;
    }
    
    /**
     * Fetch a block by its index. Returns null if not found.
     */
    public function getBlockByIndex(int $index): ?Block
    {
        return $this->chain[$index] ?? null;
    }
}
