<?php
$logFile = 'blockchain_log.txt';
$transactions = [];

if (file_exists($logFile)) {
    $lines = file($logFile, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
    foreach ($lines as $line) {
        // Expect line format: "hash | json"
        [$hash, $json] = explode(' | ', $line, 2);
        $data = json_decode($json, true);
        if ($data) {
            $data['hash'] = $hash;
            $transactions[] = $data;
        }
    }
}

function timeAgo($timestamp) {
    if (!is_numeric($timestamp)) {
        $timestamp = strtotime($timestamp);
        if ($timestamp === false) {
            return "Invalid date";
        }
    }

    $diff = time() - $timestamp;

    if ($diff < 0) {
        // Timestamp is in the future
        return 'in ' . abs($diff) . ' seconds';
    }

    if ($diff < 60) return $diff . ' sec ago';
    if ($diff < 3600) return floor($diff / 60) . ' min ago';
    if ($diff < 86400) return floor($diff / 3600) . ' hr ago';

    return date('Y-m-d H:i:s', $timestamp);
}

?>

<!DOCTYPE html>
<html lang="en">
<head>
<meta charset="UTF-8" />
<title>Blockchain Transaction Log</title>
<style>
  table { border-collapse: collapse; width: 100%; }
  th, td { border: 1px solid #ccc; padding: 8px; text-align: left; }
  .hash { font-family: monospace; font-size: 0.9em; word-break: break-all; }
</style>
<script>
  // Auto-refresh every 5 seconds
  setTimeout(() => {
    window.location.reload();
  }, 5000);
</script>
</head>
<body>

<h2>Blockchain Transaction Log</h2>

<table>
    <thead>
        <tr>
            <th>Time</th>
            <th>Action</th>
            <th>Full Name</th>
            <th>Email</th>
            <th>Reg. No</th>
            <th>Hash</th>
        </tr>
    </thead>
    <tbody>
        <?php foreach (array_reverse($transactions) as $txn): ?>
            <tr>
                <td><?= htmlspecialchars(timeAgo($txn['timestamp'] ?? '')) ?></td>
                <td><?= htmlspecialchars($txn['action'] ?? '') ?></td>
                <td>
                    <?= htmlspecialchars(
                        (($txn['details']['fname'] ?? '') . ' ' . ($txn['details']['lname'] ?? ''))
                    ) ?>
                </td>
                <td><?= htmlspecialchars($txn['details']['email'] ?? '') ?></td>
                <td><?= htmlspecialchars($txn['details']['reg_no'] ?? '') ?></td>
                <td><div class="hash"><?= htmlspecialchars($txn['hash']) ?></div></td>
            </tr>
        <?php endforeach; ?>
    </tbody>
</table>

</body>
</html>
